package com.sea.gbgs5;

import com.cinterion.io.ATCommandFailedException;
import com.cinterion.io.OutPort;
import java.io.*;
import com.sea.log.Logger;
import com.sea.modem.*;
import com.sea.utils.*;
import java.util.Enumeration;
import java.util.Timer;
import java.util.Vector;

public class Control extends Thread {

// --------------------------------------------------
// Public Properties
// --------------------------------------------------
public static LanguageAbstract language = null;
// --------------------------------------------------
// Private Constants
// --------------------------------------------------
// Machine states
private static final int CTRL_INIT = 0;
private static final int CTRL_CREATE = 10;
private static final int CTRL_START = 20;
private static final int CTRL_READY = 1000;
private static final int CTRL_STOP = 2000;
private static final int CTRL_FINISHED = 9999;

// How long sleep when ready [ms]
private static final int READY_DELAY = 10000;

private static final String FILENAME_SERIAL_NUMBER = "file:///A:/sn.txt";

// --------------------------------------------------
// Private Properties
// --------------------------------------------------
private static Control instance = null;
private static boolean bRunning = false;

// Debug Level
public static final int iLogLevel = Logger.INFO;

// Machine state
private static int iMachine;
private static boolean bWantedToStop;
private static Vector vDeviceInfo;
private static OutPort outp;

// Controlled threads
private static Console consl;
private static ConsoleReporter conslrep;
private static Slave slave;
//private static TcpReporter tcprep;
private static Watchdog wdg;

// Managed objects
// Controlled timer and its tasks
private Timer tmr1;
private Info ttInfo;

// Controlled AT channels
private static SyncAtCommand ata, atb, atc;


// --------------------------------------------------
// Class Code
// --------------------------------------------------
/**
 * Constructor
 */
public Control() {
  ata = atb = atc = null;
}

/**
 * getInstance()
 */
public static Control getInstance() {
  if (instance == null) {
    instance = new Control();
  }
  return instance;
}

/**
 * startRunning
 */
public void startRunning() {
  bWantedToStop = false;
  bRunning = true;
  this.start();
}

/**
 * stopRunning
 */
public void stopRunning() {
  if (!bWantedToStop) {
    bWantedToStop = true;
    instance.interrupt();
  }
}

/**
 * Debug printing
 */
protected static void log(String line) {
  log(line, Logger.INFO);
}

/**
 * Debug printing
 */
protected static void log(String line, int level) {
  if (iLogLevel >= level) {
    Logger.log("@Control", line);
  }
}

/**
 * run
 */
public void run() {

  log("Thread started");

  machineJumpTo(CTRL_INIT);
  while (bRunning) {
    try {
      switch (iMachine) {
      case CTRL_INIT:
        ctrlMachineInit();
        break;
      case CTRL_CREATE:
        ctrlMachineCreate();
        break;
      case CTRL_START:
        ctrlMachineStart();
        break;
      case CTRL_READY:
        ctrlMachineReady();
        break;
      case CTRL_STOP:
        ctrlMachineStop();
        break;
      case CTRL_FINISHED:
        ctrlMachineFinished();
        break;
      default:

        // inpossible state
        machineJumpTo(CTRL_INIT);
      }
    } catch (Exception e) {
      // An exception occured. Go the interrupted step again.
      log("[EXCEPTION] - " + e.toString(), Logger.ERROR);
      e.printStackTrace();
      try {
        sleep(500);
      } catch (Exception ignored) {
      }
    }

    if (bWantedToStop && bRunning) {
      machineJumpTo(CTRL_STOP);
    }

  }

  log("Thread finished.", Logger.ALWAYS);
}

/**
 * ctrlMachineInit
 */
private void ctrlMachineInit() {

  // Create AT channels
  try {
    log("Creating AT channel A");
    if (ata == null) {
      ata = new SyncAtCommand(false, "ata");
      //ata.addListener(new ControlURCListener("ata"));
    }
    log("Creating AT channel B");
    if (atb == null) {
      atb = new SyncAtCommand(true, "atb");
      //atb.addListener(new ControlURCListener("atb"));
    }
    log("Creating AT channel C");
    if (atc == null) {
      atc = new SyncAtCommand(false, "atc");
      //atc.addListener(new ControlURCListener("atc"));
    }
  } catch (ATCommandFailedException ex) {
    log("[ERROR] - Cannot create AT channels", Logger.ERROR);
  }

  selectStdout("USB");
  
  Cfg.init();
  Cfg.detectSerialNumber();
  vDeviceInfo = prepareDeviceInfo();
  printDeviceInfo(vDeviceInfo);
  
  if ((Cfg.variant == Cfg.VARIANT_GSM_MODBUS) || (Cfg.variant == Cfg.VARIANT_GSM_BGS5)) {
    enablePowerForRS485();
  }
  
  log("Comm ports available='" + System.getProperty("microedition.commports") + "'");
  log("Thread started.", Logger.ALWAYS);

  machineJumpTo(CTRL_CREATE);
}

/**
 * ctrlMachineStart
 */
private void ctrlMachineCreate() throws ATCommandFailedException, IOException {

  log("Creating Language.");
  language = new LanguageEnglish();  // ``` Udelat zavisle na konfiguraci

  log("Creating Console");
  consl = Console.getInstance();
  consl.registerAtCommand(atb);
  consl.setDeviceInfo(vDeviceInfo);
  conslrep = ConsoleReporter.getInstance();
  conslrep.registerAtCommand(atc);

  // I/O Managers
  log("Creating Watchdog.");
  wdg = Watchdog.getInstance();
  wdg.registerAtCommand(ata);
  log("Creating Slave");
  slave = Slave.getInstance();

  // Timer based tasks 
  tmr1 = new Timer();
  ttInfo = new Info();

  machineJumpTo(CTRL_START);
}

/**
 * ctrlMachineStart
 */
private void ctrlMachineStart() {
  log("Starting threads.");

  consl.startRunning();
  conslrep.startRunning();

  slave.startRunning();
  wdg.startRunning();

  tmr1.scheduleAtFixedRate(ttInfo, 0, 10000);
  /*
   // Example of TimerTask
   tmr1.schedule(new TimerTask() {
   public void run() {
   }
   }, 10000);
   */

  machineJumpTo(CTRL_READY);
}

/**
 * ctrlMachineReady
 */
private void ctrlMachineReady() {

  log(getStatus(0), Logger.ALWAYS);
  log(getStatus(1), Logger.ALWAYS);

  // Watch all other threads and restart if any of them stops working
  if (!wdg.isAlive() || !slave.isAlive()) {
    Watchdog.deviceReset();
  }

  delay(READY_DELAY);
}

/**
 * ctrlMachineStop
 */
private void ctrlMachineStop() {

  Vector threads = new Vector(); // Vector of Threads

  // --------------------------------------------------------
  // --------------------------------------------------------
  // Stop of thread Info
  log("- Timer1(Info, UserLoger, EventScheduler)");
  if (tmr1 != null) {
    tmr1.cancel();
  }

  // --------------------------------------------------------
  log("Stopping threads:");
  try {
    if (wdg != null) {
      log("- Watchdog");
      wdg.stopRunning();
      threads.addElement(wdg);
    }
  } catch (Exception ignored) {
  }

  try {
    if (slave != null) {
      log("- Slave");
      slave.stopRunning();
      threads.addElement(slave);
    }
  } catch (Exception ignored) {
  }

  // Stop of thread Console
  try {
    if (conslrep != null) {
      log("- Console Reporter");
      conslrep.stopRunning();
      threads.addElement(conslrep);
    }
  } catch (Exception ignored) {
  }

  try {
    if (consl != null) {
      log("- Console");
      consl.stopRunning();
      threads.addElement(consl);
    }
  } catch (Exception ignored) {
  }

  // --------------------------------------------------------
  log("Waiting for all threads to die ...");

  boolean bAllIsTerminated = false;
  long lStartTime = Timed.currentTimeMillis();
  while (!bAllIsTerminated && !Timed.ist(lStartTime, 10000)) {
    Thread t;
    boolean bSomethingLives = false;
    for (Enumeration e = threads.elements(); e.hasMoreElements();) {
      t = (Thread) e.nextElement();
      if (t.isAlive()) {
        bSomethingLives = true;
        log(t.getClass() + " is ALIVE");
      } else {
        //log(t.getClass() + " is dead");
      }
    }

    if (bSomethingLives) {
      log("  wait ...");
      try {
        sleep(300);
      } catch (InterruptedException ex) {
      }
    } else {
      bAllIsTerminated = true;
    }

  }

  // --------------------------------------------------------
  log("Releasing AT Channels");
  log("- ATB");
  try {
    atb.release();
  } catch (Exception ex) {
  }

  log("- ATC1");
  selectStdout("USB1");  // Switch stdout to safe direction
  log("- ATC2");
  try {
    atc.release();
  } catch (Exception ex) {
  }

  log("- ATA");
  try {
    ata.release();
  } catch (Exception ex) {
  }

  log("Stop completed.");
  bRunning = false;

  machineJumpTo(CTRL_FINISHED);
}

/**
 * ctrlMachineFinished
 */
private void ctrlMachineFinished() {
  bRunning = false;
}

/**
 * delay
 *
 * @param millis long
 */
private void delay(long millis) {
  try {
    sleep(millis);
  } catch (InterruptedException ignored) {
  }
}

// Jump to named state of machine (dtMachineStateH)
private void machineJumpTo(int aNewState) {
  iMachine = aNewState;
}

/**
 * printDeviceInfo()
 *
 * @param src
 */
public static void printDeviceInfo(Vector src) {

  for (int i = 0; i < src.size(); i++) {
    log((String) src.elementAt(i), Logger.ALWAYS);
  }

}

/**
 * prepareDeviceInfo()
 *
 * @return
 */
public static Vector prepareDeviceInfo() {

  Vector result = new Vector(7);
  result.addElement("--- START --------------------------------------------------------");
  result.addElement("Product:       " + Main.PRODUCT_NAME + " (" + Main.PRODUCT_NUMBER + ")");
  result.addElement("Version:       " + Main.PRODUCT_VERSION);
  result.addElement("Manufacturer:  SEA s.r.o., Czech Republic");
  result.addElement("               www.seapraha.cz");
  result.addElement("Serial Number: " + formatSerialNumber());
  result.addElement("------------------------------------------------------------------");
  return result;
}

/**
 * detectSerialNumber()
 */
private void detectSerialNumber() {

  String sText;
  LineReader lReader;
  String s;
  String sResult = "";

  // Read whole file into a String
  sText = FileUtils.readTextFile(FILENAME_SERIAL_NUMBER);
  lReader = new LineReader(sText);
  // Get the first line
  s = lReader.getLine();
  if (s != null) {
    if (s.length() >= 3) {
      if (s.charAt(0) == 'G') {
        sResult = s.trim();
      }
    }
  }

  // Default serial number for exceptional casea (because SeaConfigurator will get mad without serial number)
  if (sResult.length() == 0) {
    sResult = "GC914001";
  }
  Cfg.SERIAL_NUMBER = sResult;

}

/**
 * printSerialNumber()
 *
 * @return String
 */
private static String formatSerialNumber() {

  if (Cfg.SERIAL_NUMBER.length() > 0) {
    return Cfg.SERIAL_NUMBER;
  } else {
    return "-";
  }

}

/**
 * getStatus
 *
 * @return String
 */
private String getStatus(int _type) {
  String result = "";

  if (_type == 0) {
    result += "A";

  } else {
    result += "  "
            + Main.PRODUCT_NAME + " " + Main.PRODUCT_VERSION + " " + Cfg.SERIAL_NUMBER;
  }

  return result;
}

/**
 * Set STDOUT to ASC1
 *
 * @param destination "ASC0", "ASC1" or "NULL", "USB1"
 */
public void selectStdout(String destination) {
  try {
    atc.send("at^scfg=\"Userware/Stdout\",\"" + destination + "\"\r");
  } catch (ATCommandFailedException ignored) {
  }
}

static boolean isDeviceGsmKlic() {
  return Cfg.SERIAL_NUMBER.startsWith("GD1");
}

static boolean isDeviceGsmR2Ring() {
  return Cfg.SERIAL_NUMBER.startsWith("GD3");
}

static boolean isDeviceGsmR2T() {
  return Cfg.SERIAL_NUMBER.startsWith("GD2");
}

static boolean isDeviceGsmPress() {
  return Cfg.SERIAL_NUMBER.startsWith("GD7");
}

static boolean isDeviceGsmBgs5() {
  return Cfg.SERIAL_NUMBER.startsWith("GD5");
}

static boolean isDeviceGsmCmgsm() {
  return Cfg.SERIAL_NUMBER.startsWith("GD4");
}

static boolean isDeviceGsmR5MODBUS() {
  return Cfg.SERIAL_NUMBER.startsWith("GF8"); //``` deresit zbytel R5 rodiny
}

static boolean isDeviceGsmR5T() {
  return Cfg.SERIAL_NUMBER.startsWith("GF4");
}


/**
 * Enable power supply for GBGS5M (if present) - GPIO23 must be an output and H
 */
private void enablePowerForRS485() {
  try {
    atc.send("at^scpin=1,22,1,1\r");
  } catch (ATCommandFailedException ex) {
    ex.printStackTrace();
  }
}

}
