/*
 * Working with files

 */
package com.sea.utils;

import com.sea.log.Logger;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import javax.microedition.io.Connector;
import javax.microedition.io.file.FileConnection;

/**
 *
 * @author vorisek
 */
public class FileUtils {

// --------------------------------------------------
// Public Constants
// --------------------------------------------------
// --------------------------------------------------
// Public Properties
// --------------------------------------------------
public static int iLogLevel = Logger.WARN;

// --------------------------------------------------
// Private Constants
// --------------------------------------------------
private static final String FILENAME_ROOT = "file:///A:/";
// --------------------------------------------------
// Private Properties
// --------------------------------------------------
// --------------------------------------------------
// Public Methods
// --------------------------------------------------

/**
 * Read whole file into a String
 * 
 * @param sFilename, example "file:///A:/ADFiles/calibration.xml"
 * @return Whole content of the file or empty string or empty string if error occurs (i.e. file doesn't exists)
 */
public static String readTextFile(String sFilename) {

  InputStream is = null;
  FileConnection fc = null;
  String result = "";

  try {
    log("Readfing file '" + sFilename + "'.");
    fc = (FileConnection) Connector.open(sFilename, Connector.READ_WRITE);

    if (fc.exists()) {
      int size = (int) fc.fileSize();
      is = fc.openInputStream();
      byte bytes[] = new byte[size];
      is.read(bytes, 0, size);
      result = new String(bytes, 0, size);
      log("Content '" + getPreview(result, 400) + "'.");
    }
  } catch (IOException ioe) {
    // There is an error while opening or reading the file. Return empty string.
    log("[EXCEPTION] - " + ioe.getMessage(), Logger.ERROR);
  } finally {
    try {
      if (null != is) {
        is.close();
      }
      if (null != fc) {
        fc.close();
      }
    } catch (IOException e) {
      System.out.println(e.getMessage());
    }
  }
  return result;
}

/**
 * Write whole file at once
 */
public static void writeTextFile(String sFilename, String sText) {
  OutputStream os = null;
  FileConnection fconn = null;
  try {
    log("Writing file '" + sFilename + "' with content '" + getPreview(sText, 400) + "'.");
    fconn = (FileConnection) Connector.open(sFilename, Connector.READ_WRITE);
    if (!fconn.exists()) {
      fconn.create();
    }

    os = fconn.openOutputStream();
    os.write(sText.getBytes());
    fconn.setHidden(false);
//          fconn.setReadable(true);
  } catch (IOException e) {
    log("[EXCEPTION] - " + e.getMessage(), Logger.ERROR);
  } finally {
    try {
      if (null != os) {
        os.close();
      }
      if (null != fconn) {
        fconn.close();
      }
    } catch (IOException e) {
      System.out.println(e.getMessage());
    }
  }
}

/**
 * Delete a file
 * 
 * @param sFilename 
 * @return
 */
public static boolean delete(String sFilename) {

  FileConnection fin;
  boolean result = false;
  //log("Detele file '" + sFilename + "' ... ");
  try {
    fin = (FileConnection) Connector.open(sFilename);
    if (fin.exists()) {
      fin.delete();
      //log("[  OK  ] - deleted");
      result = true;
    } else {
      //log("[  OK  ] - does not exists");
    }
    fin.close();

  } catch (IOException e) {
    log("[EXCEPTION] - FileUtils.detele('" + sFilename + "'): " + e.getMessage(), Logger.ERROR);
  }
  return result;
}

/**
 * Ensure that directory exists
 *
 * @param sPath path to directory, i.e. "file:///A:/logger"
 */
public static void force(String sPath) {

  FileConnection fconn = null;
  if (sPath.charAt(sPath.length() - 1) == '/') {
    sPath = sPath.substring(0, sPath.length() - 1);
  }

  try {
    fconn = (FileConnection) Connector.open(sPath, Connector.READ_WRITE);
    if (!fconn.isDirectory()) {
      log("Creating '" + sPath + "' because it did not exists.");
      fconn.mkdir();
    }

  } catch (IOException e) {
    log("[EXCEPTION] - " + e.getMessage(), Logger.ERROR);
  } finally {
    try {
      if (null != fconn) {
        fconn.close();
      }
    } catch (IOException e) {
      System.out.println(e.getMessage());
    }
  }
}

/**
 * Open, read and close file
 *
 * @param sFilename
 * @param iOffset
 * @param iLength
 * @return
 */
public static byte[] readFile(String sFilename, long iOffset, int iLength) {

  InputStream is = null;
  FileConnection fc = null;
  byte[] result = new byte[0];

  try {
    //log("Readfing file '" + sFilename + "'.");
    fc = (FileConnection) Connector.open(sFilename, Connector.READ_WRITE);

    if (fc.exists()) {
      int size = (int) fc.fileSize();
      if (iOffset >= size) {
        return new byte[0];
      }
      is = fc.openInputStream();
      is.skip(iOffset);
      long remainingSize = size - iOffset;
      long correctedLength = (iLength > remainingSize) ? remainingSize : iLength;
      byte bytes[] = new byte[(int) correctedLength];
      is.read(bytes, 0, bytes.length);
      result = bytes;
      //log("Content '" + getPreview(result, 400) + "'.");
    }
  } catch (IOException ioe) {
    // There is an error while opening or reading the file. Return empty string.
    log("[EXCEPTION] - " + ioe.getMessage(), Logger.ERROR);
  } finally {
    try {
      if (null != is) {
        is.close();
      }
      if (null != fc) {
        fc.close();
      }
    } catch (IOException e) {
      System.out.println(e.getMessage());
    }
  }
  return result;
}

/**
 * Return size of a file in bytes
 *
 * @param sFilename
 * @return
 */
public static int fileSize(String sFilename) {

  FileConnection fc = null;
  int result = 0;

  try {
    //log("Opening file '" + sFilename + "'.");
    fc = (FileConnection) Connector.open(sFilename, Connector.READ_WRITE);

    if (fc.exists()) {
      result = (int) fc.fileSize();
  }
  } catch (IOException ioe) {
    // There is an error while opening or reading the file. Return empty string.
    log("[EXCEPTION] - " + ioe.getMessage(), Logger.ERROR);
  } finally {
    try {
      if (null != fc) {
        fc.close();
      }
    } catch (IOException e) {
      System.out.println(e.getMessage());
    }
  }
  return result;
}

/**
 *
 * @param filename
 * @return
 */
static public boolean exists(String filename) {

  FileConnection fc = null;
  boolean result = false;

  try {
    fc = (FileConnection) Connector.open(filename, Connector.READ_WRITE);
    result = fc.exists();
  } catch (IOException ex) {
    log("[ERROR] - Cannot access file '" + filename + "' ex=" + ex.getMessage());
  } finally {
    try {
      if (null != fc) {
        fc.close();
      }
    } catch (IOException e) {
      log(e.getMessage());
    }
  }
  return result;
}

// --------------------------------------------------
// Private Methods
// --------------------------------------------------
/**
 * Debug printing
 */
private static void log(String line) {
  log(line, Logger.INFO);
}

/**
 * Debug printing
 */
private static void log(String line, int level) {
  if (iLogLevel >= level) {
    Logger.log("@File", line);
  }
}

/**
 * Return first max. 'len' character from String 'src'
 * 
 * @param src
 * @param len
 * @return String
 */
private static String getPreview(String src, int len) {
  if (len < src.length()) {
    return src.substring(0, len);
  } else {
    return src;
  }
}

}
