package com.sea.utils;

import com.sea.log.Logger;
import com.sea.modem.SyncAtCommand;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * General software watchdog
 *
 * @author Vorisek
 */
public class Watchdog extends Thread {

// --------------------------------------------------
// Public Properties
// --------------------------------------------------
// --------------------------------------------------
// Private Constants
// --------------------------------------------------
private long TIMEOUT_ALL_RESPONDING = 15 * 1000; // [sec]

// --------------------------------------------------
// Private Properties
// --------------------------------------------------
private static Watchdog instance = null;
private static SyncAtCommand atc = null;
private boolean bRunning = false;

// Debug Level
public static final int iLogLevel = Logger.INFO;

// Clients
private Hashtable clients;

/**
 * Constructor
 */
public Watchdog() {
  super();
  clients = new Hashtable();
}

/**
 * getInstance()
 *
 * @return
 */
public static Watchdog getInstance() {
  if (instance == null) {
    instance = new Watchdog();
  }
  return instance;
}

/**
 * Register an ATCommand for use with this singleton instance.
 *
 * @param aSyncAtCommand ATCommand to use.
 */
public void registerAtCommand(SyncAtCommand aSyncAtCommand) {
  atc = aSyncAtCommand;
}

/**
 * startRunning
 */
public void startRunning() {
  bRunning = true;
  this.start();
}

/**
 * stopRunning
 */
public void stopRunning() {
  bRunning = false;
  instance.interrupt();
}

/**
 * Debug printing
 */
protected static void log(String line) {
  log(line, Logger.INFO);
}

/**
 * Debug printing
 */
protected static void log(String line, int level) {
  if (iLogLevel >= level) {
    Logger.log("@Watchdog", line);
  }
}

public void run() {

  WatchdogClient wc;
  long iNow;
  boolean bAllResponding;
  long iLastAllReponding = Timed.currentTimeMillis();

  while (bRunning) {
    try {

      sleep(1000);

      iNow = Timed.currentTimeMillis();
      bAllResponding = true;
      for (Enumeration e = clients.elements(); e.hasMoreElements();) {
        wc = (WatchdogClient) e.nextElement();
        if (wc.ist(iNow)) {
          bAllResponding = false;
          log("[ERROR] - Client is not responding: " + wc.getName());
        }
      }

      if (bAllResponding) {
        iLastAllReponding = iNow;
      } else {
        if ((iNow - iLastAllReponding) >= TIMEOUT_ALL_RESPONDING) {
          log("[ERROR] - Patience is over --> RESET");
          deviceReset();
        }
      }

    } catch (InterruptedException ignored) {
    } catch (Exception e) {
      log("[EXCEPTION] - " + e.getMessage());
      e.printStackTrace();
    }
  }

  log("Thread finished");

  instance = null;
}

/**
 * Register a client to watchdog service
 *
 * @param client
 */
public void register(Object client) {
  clients.put(client, new WatchdogClient(client.getClass().getName()));

}

/**
 * Refresh timer for a client
 *
 * @param client
 * @param nextLimit [sec]
 */
public void kick(Object client, int nextLimit) {

  if (this.isAlive()) {
    try {
      WatchdogClient wc;
      wc = (WatchdogClient) clients.get(client);
      wc.setNext(nextLimit);
    } catch (Exception e) {
      log("[EXCEPTION] - Unregisted client tries to use Watchdog, e - " + e.getMessage());
      e.printStackTrace();
      deviceReset();
    }
  }
}

/**
 * Immediate restart
 */
public static void deviceReset() {
  (new Thread(new WatchdogDeviceReset(atc))).start();
}

// --------------------------------------------------
// Private class
// --------------------------------------------------
private class WatchdogClient {

private long iNext; // [ms]
private String sName;

public WatchdogClient(String name) {
  iNext = 0; // 0=not active
  sName = name;
}

/**
 *
 * @param iNow
 * @return
 */
public boolean ist(long iNow) {
  return ((iNow >= iNext) && (iNext != 0));
}

public void setNext(long value) {
  iNext = Timed.currentTimeMillis() + (value * 1000);
}

public String getName() {
  return sName;
}
}

}
