/**
 * Console for testing/debugging over serial port
 */
package com.sea.gbgs5;

import com.sea.log.Logger;
import com.sea.modem.SyncAtCommand;
import com.sea.utils.FileUtils;
import com.sea.utils.Utils;
import java.io.*;
import java.util.Vector;
import javax.microedition.io.*;
import javax.microedition.io.file.FileConnection;

/**
 *
 * @author vorisek
 */
public class Console extends Thread {
// --------------------------------------------------
// Public Properties
// --------------------------------------------------
// --------------------------------------------------
// Private Constants
// --------------------------------------------------

private static final String CONNECTION_STRING = "comm:USB0;baudrate=115200;autocts=on;autorts=on";
private static final String FILEPATH_CALIBRATION = "file:///A:/factory_calibration/";

// --------------------------------------------------
// Private Properties
// --------------------------------------------------
private static Console instance;
private boolean bRunning;
private SyncAtCommand atc;
public static final int iLogLevel = Logger.INFO;
private Vector vDeviceInfo;

private InputStream ins;
//private OutputStream outs;
//private PrintStream out;

private CommConnection commConn;

// --------------------------------------------------
// Class Code
// --------------------------------------------------
public Console() {
  super();
}

/**
 * getInstance
 */
public static Console getInstance() {
  if (instance == null) {
    instance = new Console();
  }
  return instance;
}

/**
 * startRunning
 */
public void startRunning() {
  bRunning = true;
  this.start();
}

/**
 * stopRunning
 */
public void stopRunning() {
  bRunning = false;
  this.interrupt();
}

/**
 * Debug printing
 */
protected static void log(String line) {
  log(line, Logger.INFO);
}

/**
 * Debug printing
 */
protected static void log(String line, int level) {
  if (iLogLevel >= level) {
    Logger.log("@Console", line);
  }
}

/**
 * Register an ATCommand for use with this singleton instance.
 *
 * @param aSyncAtCommand
 */
public void registerAtCommand(SyncAtCommand aSyncAtCommand) {
  atc = aSyncAtCommand;
}

/**
 * run
 */
public void run() {

  String line;
  String s;
  String result;
  boolean bAuthorized = false;
  int index;
  int value;
  Vector parts;

  log("Thread started", Logger.ALWAYS);
  commConn = null;
  while (bRunning) {
    try {

      if (commConn == null) {
        log("Opening connection for Console - " + CONNECTION_STRING);
        openPort();
      }

      if (!bAuthorized) {
        System.out.println("Enter 'console'<cr> to enter command mode.");
      }

      line = readln(ins);

      if (!bAuthorized) {
        if (line.equalsIgnoreCase("console")) {
          bAuthorized = true;
          line = "-";
        }
      }

      if (bAuthorized) {
        char cc = (line.length() >= 1) ? line.charAt(0) : '-';
        cc = Character.toLowerCase(cc);

        switch (cc) {

        case 'a':
          if ((line.startsWith("AT")) || (line.startsWith("at"))) {
            String response = atc.send(line + "\r");
            System.out.println(response);
          } else {
            System.out.println("ERROR - AT command must begin with 'at'");
          }
          break;

        case 'c':
          Cfg.printConfiguration();
          break;

        case 'd':
          if (line.length() > 1) {
            s = line.substring(1);
            value = Integer.parseInt(s);
          } else {
            value = 0;
          }
          if (value == 0) {
            ConsoleReporter.getInstance().reportOnceAndPause();
          } else {
            ConsoleReporter.getInstance().resume(value);
          }
          break;

        case 'f':
          s = line.substring(1);
          parts = Utils.splitComma(s);
          forceCalibrationPathExists(FILEPATH_CALIBRATION);
          if (parts.isEmpty()) {
            // read all
            index = 1; // Now calibration for A1 only
            s = FileUtils.readTextFile(FILEPATH_CALIBRATION + index + ".ini");
            if (s.length() != 0) {
              System.out.println(s);
            } else {
              System.out.println("<none>");
            }
            System.out.println("OK");
          } else if (parts.size() >= 1) {
            // erase or write
            index = Integer.parseInt(((String) parts.elementAt(0)));
            if (parts.size() >= 2) {
              // write
              FileUtils.writeTextFile(FILEPATH_CALIBRATION + index + ".ini", s);
              System.out.println("OK - write calibration for A" + index);
            } else {
              // erase
              FileUtils.delete(FILEPATH_CALIBRATION + index + ".ini");
              System.out.println("OK - erase calibration for A" + index);
            }
          }

          break;

        case 'i':
          for (int i = 0; i < vDeviceInfo.size(); i++) {
            System.out.println((String) vDeviceInfo.elementAt(i));
          }
          break;

        case 'l':
          s = line.substring(1);
          parts = Utils.splitComma(s);
          index = Integer.parseInt(((String) parts.elementAt(0)));
          if (parts.size() > 1) {
            value = Integer.parseInt(((String) parts.elementAt(1)));
          } else {
            value = -1;
          }
          if (index == 1) {
            if (value == -1) {
              Slave.getInstance().overrideLedOut(false, Slave.LD_AUTO);
            } else {
              Slave.getInstance().overrideLedOut(true, value);
            }
            result = "OK";
          } else if (index == 2) {
            if (value == -1) {
              Slave.getInstance().overrideLedGsm(false, Slave.LD_AUTO);
            } else {
              Slave.getInstance().overrideLedGsm(true, value);
            }
            result = "OK";
          } else {
            result = "ERROR - invalid index, available 1 (OUT) and 2 (GSM)";
          }
          System.out.println(result);
          break;

        case 'o':
          s = line.substring(1);
          parts = Utils.splitComma(s);
          index = Integer.parseInt(((String) parts.elementAt(0)));
          if ((index == 0) || (index == 2)) {
            value = Integer.parseInt(((String) parts.elementAt(1)));
            result = Slave.getInstance().setOut(0, value);
          } else {
            result = "ERROR - invalid index, available 0 and 2";
          }
          System.out.println(result);
          break;

        case 'p':
          if (line.length() > 1) {
            s = line.substring(1);
            value = Integer.parseInt(s);
          } else {
            value = 0;
          }
          if (value == 0) {
            Slave.getInstance().reportOnceAndPause();
          } else {
            Slave.getInstance().resume(value);
          }
          break;

        case 'q':
          System.out.println("Quit JAVA application.");
          bRunning = false;
          stopMidlet();
          break;

        case 'w':
          System.out.println("Sending command to HW Watchdog to reset ...");
          Slave.getInstance().forceHwReset();
          break;
          
        case 'x':
          System.out.println("Entering sleep mode ...");
          Slave.getInstance().sleep();
          break;

        case '#':
          bAuthorized = false;
          Slave slave = Slave.getInstance();
          slave.overrideLedOut(false, Slave.LD_AUTO);
          slave.overrideLedGsm(false, Slave.LD_AUTO);
          break;

        default:
          // Print help
          System.out.println("--- HELP ------------------------------------------------");
          System.out.println("a .. AT command, at+creg?");
          System.out.println("c .. print configuration");
          System.out.println("d .. print state, d1=start auto gpio info");
          sleep(100);
          System.out.println("f .. calibration write: f1,a,\"2015-12-14 15:13:00\",25.5,548"); // 1=idx A1, a=verze, 25.5='C, 548=ADC
          System.out.println("                 erase: f1"); // 1=idx A1
          System.out.println("                 read:  f");
          System.out.println("i .. system info");
          sleep(100);
          System.out.println("l .. LED control: l1,0 = LED OUT on; l1,0 = LED OUT off; l1=OUT; l2=GSM");
          System.out.println("o .. swith output: o2,0   o2,1");
          System.out.println("p .. print press debug info, p1=start, p0=once and stop ");
          System.out.println("q .. switch off Java autostart and restart");
          sleep(100);
          System.out.println("w .. force HW watchdog to reset device");
          System.out.println("x .. enter sleep mode");
          System.out.println("# .. leave commmand mode");
          System.out.println("---------------------------------------------------------");
        }
      }
      //} catch (InterruptedException ignored) {
    } catch (Exception e) {
      // An exception occured. Go the interrupted step again.
      log("[ EXCEPTION ] - '" + e.toString() + "'", Logger.ERROR);
      e.printStackTrace();

    }
  }

  closePort();
  log("Thread finished", Logger.ALWAYS);

}

/**
 * openPort()
 */
private void openPort() throws IOException {
  // Open ASC port to configurator
  commConn = (CommConnection) Connector.open(CONNECTION_STRING, Connector.READ_WRITE, true);
  ins = commConn.openInputStream();
//  outs = commConn.openDataOutputStream();
//  out = new PrintStream(outs);

}

/**
 * closeSlaveConnection()
 */
private void closePort() {
  /*  out.close();
   if (outs != null) {
   try {
   outs.flush();
   outs.close();
   outs = null;
   } catch (Exception ignored) {
   }
   }*/
  if (ins != null) {
    try {
      ins.close();
      ins = null;
    } catch (Exception ignored) {
    }
  }
  if (commConn != null) {
    try {
      commConn.close();
      commConn = null;
    } catch (Exception ignored) {
    }
  }
}

/**
 * Read line from
 *
 * @param dataInputStream DataInputStream
 * @return String
 */
private String readln(InputStream is) {

  final int MAX_LINE_LENGTH = 256;

  int pos = 0;
  byte aLine[] = new byte[MAX_LINE_LENGTH];
  int c;

  try {
    while ((c = is.read()) >= 0) {
      if ((c == '\r') || (c == '\n')) {
        break; // CR of LF will break this loop
      }
      if (pos < MAX_LINE_LENGTH) {
        aLine[pos++] = (byte) c; // Save received character if space is available
      }
    }
  } catch (IOException ignored) {
  }

  return new String(aLine, 0, pos);
}

/**
 * setDeviceInfo()
 *
 * @param vDeviceInfo
 */
void setDeviceInfo(Vector src) {
  vDeviceInfo = src;
}

private void forceCalibrationPathExists(String sPath) {

  FileConnection fconn = null;
  if (sPath.charAt(sPath.length() - 1) == '/') {
    sPath = sPath.substring(0, sPath.length() - 1);
  }

  try {
    fconn = (FileConnection) Connector.open(sPath, Connector.READ_WRITE);
    if (!fconn.isDirectory()) {
      fconn.mkdir();
    }

  } catch (IOException e) {
    log("[EXCEPTION] - " + e.getMessage(), Logger.ERROR);
  } finally {
    try {
      if (null != fconn) {
        fconn.close();
      }
    } catch (IOException e) {
      System.out.println(e.getMessage());
    }
  }
}

private void stopMidlet() {
  (new Thread() {
  public void run() {
    Main.getInstance().destroyApp(true);
  }
  }).start();
}

}
